/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.Area;
import com.inet.report.CrossTab;
import com.inet.report.CrossTabBody;
import com.inet.report.CrossTabBodyCell;
import com.inet.report.CrossTabHeaderList;
import com.inet.report.DatabaseField;
import com.inet.report.Engine;
import com.inet.report.FieldElement;
import com.inet.report.Fields;
import com.inet.report.ReportProperties;
import com.inet.report.Section;
import com.inet.report.SummaryField;

/**
 * This sample shows the usage of crosstabs with RDC. RDC does not support the creation of database connections, so we
 * have to use an existing report, that contains some database fields. As soon as RDC supports database connections
 * created by your own, delete the first section and fill in your database fields directly.
 */
public class CrossTabSample extends RDCSample {

    // Change this before running the example if the location of the Report
    // "mainrep.rpt" which comes with the rdc-samples has changed
    static private final String REPORTFILE = "samples/rdc/crosstab.rpt";

    /**
     * Add a crosstab to report header section of an existing report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = new Engine( exportFmt );
            //set the name of the report file, that contain
            URL reportURL = getClass().getResource( '/' + REPORTFILE );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( REPORTFILE );
            }
            //set paper orientation and paper size, if you don't want the default (locale dependent).
            eng.getReportProperties().setPaperOrient( ReportProperties.LANDSCAPE, ReportProperties.PAPER_A4 );

            //get the fields set from engine
            Fields fields = eng.getFields();
            //get the first two database fields
            DatabaseField dbField0 = fields.getDatabaseField( 1 );
            DatabaseField dbField1 = fields.getDatabaseField( 0 );

            //get the report header area - alternative use: eng.getArea(0);
            Area area = eng.getArea( "RH" );
            //get the first section of this area
            Section section = area.getSection( 0 );
            //add a CrossTab element to section on position (0,0)
            //and dbField0 in cross table cell
            CrossTab ct = section.addCrossTab( 0, 0, null );
            CrossTabBody body = ct.getBody();
            CrossTabHeaderList cols = ct.getColumns();
            CrossTabHeaderList rows = ct.getRows();

            //add summary fields to the crosstab.
            body.addSummaryField( dbField0, SummaryField.SUM );
            SummaryField sumField1 = body.addSummaryField( dbField1, SummaryField.COUNT );

            //add a column based on dbField0 to cross table
            cols.add( dbField0 );
            //add a row based on dbField1 to cross table
            rows.add( dbField1 );

            // change the background color and font of row#1 
            rows.get( 1 ).getFieldElement().setBackColor( 0x10705C );
            FieldElement f = rows.get( 1 ).getFieldElement();
            f.setItalic( true );
            f.setFontSize( 16 );

            CrossTabBodyCell ctCell;
            //give these FieldElements another color
            for( int i = 0; i < body.getCellCount(); i++ ) {
                ctCell = body.getCell( i );
                FieldElement fElem = ctCell.getFieldElement( 0 );
                fElem.setFontColor( 0x00FF0000 );
            }

            //give these FieldElements another color
            for( int i = 0; i < body.getCellCount(); i++ ) {
                ctCell = body.getCell( i );
                FieldElement fElem = ctCell.getFieldElement( 1 );
                fElem.setFontColor( 0x000000FF );
            }

            eng.setSF( "{Customers.CustomerID} < 12315" );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new CrossTabSample().initUI();
    }
}
